﻿<#
.SYNOPSIS
  Localhost Connection Repair Tool für Windows 11
.DESCRIPTION
    Behebt localhost-Verbindungsprobleme nach Windows Updates KB5066835 und KB5065789
    durch Deaktivierung von HTTP/2 oder Deinstallation der Updates.
.PARAMETER language
.NOTES
  Version:        1.0
  Author:         Jörn Walter IT-Service Walter
  Creation Date:  2025-10-019
#>

# Prüfe Administrator-Rechte und starte neu falls erforderlich
$currentPrincipal = New-Object Security.Principal.WindowsPrincipal([Security.Principal.WindowsIdentity]::GetCurrent())
$isAdmin = $currentPrincipal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)

if (-not $isAdmin) {
    # Script-Pfad ermitteln
    $scriptPath = $MyInvocation.MyCommand.Path
    
    if ($scriptPath) {
        try {
            # Informationsdialog (optional)
            Add-Type -AssemblyName System.Windows.Forms
            $result = [System.Windows.Forms.MessageBox]::Show(
                "Dieses Tool benötigt Administrator-Rechte.`n`nMöchten Sie das Tool jetzt als Administrator neu starten?",
                "Administrator-Rechte erforderlich",
                [System.Windows.Forms.MessageBoxButtons]::YesNo,
                [System.Windows.Forms.MessageBoxIcon]::Question
            )
            
            if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
                # Script mit Admin-Rechten neu starten
                Start-Process -FilePath "powershell.exe" -ArgumentList "-NoProfile -ExecutionPolicy Bypass -File `"$scriptPath`"" -Verb RunAs
            }
        }
        catch {
            # Fallback ohne Dialog
            Start-Process -FilePath "powershell.exe" -ArgumentList "-NoProfile -ExecutionPolicy Bypass -File `"$scriptPath`"" -Verb RunAs
        }
    }
    
    # Aktuelles Script beenden
    exit
}

Add-Type -AssemblyName PresentationFramework, PresentationCore, WindowsBase

# XAML Definition für moderne GUI
$xaml = @"
<Window xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
        xmlns:x="http://schemas.microsoft.com/winfx/2006/xaml"
        Title="Localhost Connection Repair Tool" 
        Height="720" Width="650"
        WindowStartupLocation="CenterScreen"
        ResizeMode="NoResize"
        Background="#F5F5F5">
    
    <Window.Resources>
        <Style TargetType="Button">
            <Setter Property="Background" Value="#0078D4"/>
            <Setter Property="Foreground" Value="White"/>
            <Setter Property="BorderThickness" Value="0"/>
            <Setter Property="Padding" Value="15,8"/>
            <Setter Property="FontSize" Value="14"/>
            <Setter Property="Cursor" Value="Hand"/>
            <Setter Property="Template">
                <Setter.Value>
                    <ControlTemplate TargetType="Button">
                        <Border Background="{TemplateBinding Background}" 
                                CornerRadius="4"
                                Padding="{TemplateBinding Padding}">
                            <ContentPresenter HorizontalAlignment="Center" 
                                            VerticalAlignment="Center"/>
                        </Border>
                    </ControlTemplate>
                </Setter.Value>
            </Setter>
            <Style.Triggers>
                <Trigger Property="IsMouseOver" Value="True">
                    <Setter Property="Background" Value="#005A9E"/>
                </Trigger>
                <Trigger Property="IsEnabled" Value="False">
                    <Setter Property="Background" Value="#CCCCCC"/>
                    <Setter Property="Cursor" Value="Arrow"/>
                </Trigger>
            </Style.Triggers>
        </Style>
        
        <Style x:Key="DangerButton" TargetType="Button" BasedOn="{StaticResource {x:Type Button}}">
            <Setter Property="Background" Value="#D13438"/>
            <Style.Triggers>
                <Trigger Property="IsMouseOver" Value="True">
                    <Setter Property="Background" Value="#A71D2A"/>
                </Trigger>
            </Style.Triggers>
        </Style>
    </Window.Resources>
    
    <Grid Margin="20">
        <Grid.RowDefinitions>
            <RowDefinition Height="Auto"/>
            <RowDefinition Height="Auto"/>
            <RowDefinition Height="*"/>
            <RowDefinition Height="Auto"/>
        </Grid.RowDefinitions>
        
        <!-- Header -->
        <Border Grid.Row="0" Background="White" CornerRadius="8" Padding="20" Margin="0,0,0,15">
            <StackPanel>
                <TextBlock Text="🔧 Localhost Verbindungs-Reparatur" 
                          FontSize="20" 
                          FontWeight="Bold"
                          Foreground="#333333"
                          Margin="0,0,0,10"/>
                <TextBlock Text="Behebt Probleme mit localhost nach Windows 11 Updates KB5066835 und KB5065789" 
                          FontSize="12" 
                          Foreground="#666666"
                          TextWrapping="Wrap"/>
            </StackPanel>
        </Border>
        
        <!-- Optionen -->
        <Border Grid.Row="1" Background="White" CornerRadius="8" Padding="20" Margin="0,0,0,15">
            <StackPanel>
                <TextBlock Text="Reparatur-Optionen:" 
                          FontSize="14" 
                          FontWeight="Bold"
                          Foreground="#333333"
                          Margin="0,0,0,10"/>
                
                <Border Background="#FFF4E5" 
                        CornerRadius="4" 
                        Padding="12" 
                        Margin="0,0,0,10">
                    <StackPanel>
                        <TextBlock Text="Option 1: HTTP/2 deaktivieren (Registry)" 
                                  FontWeight="SemiBold"
                                  Foreground="#333333"
                                  Margin="0,0,0,5"/>
                        <TextBlock Text="Deaktiviert HTTP/2 über Registry-Einträge. Schnelle Lösung, aber HTTP/2 bleibt deaktiviert."
                                  FontSize="11"
                                  Foreground="#666666"
                                  TextWrapping="Wrap"
                                  Margin="0,0,0,8"/>
                        <Button Name="btnDisableHttp2" Content="HTTP/2 deaktivieren" Width="200" HorizontalAlignment="Left"/>
                    </StackPanel>
                </Border>
                
                <Border Background="#FFE5E5" 
                        CornerRadius="4" 
                        Padding="12">
                    <StackPanel>
                        <TextBlock Text="Option 2: Updates deinstallieren (Empfohlen)" 
                                  FontWeight="SemiBold"
                                  Foreground="#333333"
                                  Margin="0,0,0,5"/>
                        <TextBlock Text="Entfernt die problematischen Updates vollständig. Sicherste Lösung mit Neustart erforderlich."
                                  FontSize="11"
                                  Foreground="#666666"
                                  TextWrapping="Wrap"
                                  Margin="0,0,0,8"/>
                        <TextBlock Name="txtUpdateStatus"
                                  FontSize="11"
                                  FontWeight="SemiBold"
                                  Margin="0,0,0,8"
                                  TextWrapping="Wrap"/>
                        <Button Name="btnUninstallUpdates" 
                               Content="Updates deinstallieren" 
                               Style="{StaticResource DangerButton}"
                               Width="200" 
                               HorizontalAlignment="Left"/>
                    </StackPanel>
                </Border>
            </StackPanel>
        </Border>
        
        <!-- Status Log -->
        <Border Grid.Row="2" Background="White" CornerRadius="8" Padding="15">
            <ScrollViewer VerticalScrollBarVisibility="Auto">
                <TextBox Name="txtLog" 
                        IsReadOnly="True"
                        BorderThickness="0"
                        Background="Transparent"
                        TextWrapping="Wrap"
                        FontFamily="Consolas"
                        FontSize="11"
                        Foreground="#333333"/>
            </ScrollViewer>
        </Border>
        
        <!-- Footer -->
        <Grid Grid.Row="3" Margin="0,15,0,0">
            <Grid.ColumnDefinitions>
                <ColumnDefinition Width="*"/>
                <ColumnDefinition Width="Auto"/>
            </Grid.ColumnDefinitions>
            
            <!-- Copyright -->
            <TextBlock Grid.Column="0" VerticalAlignment="Center">
                <Run Text="© 2025 Jörn Walter IT-Service Walter" FontSize="11" Foreground="#666666"/>
                <LineBreak/>
                <Hyperlink Name="linkWebsite" 
                          NavigateUri="https://www.it-service-walter.com"
                          TextDecorations="None"
                          Foreground="#0078D4">
                    <Run Text="www.it-service-walter.com" FontSize="11"/>
                </Hyperlink>
            </TextBlock>
            
            <!-- Schließen Button -->
            <Button Grid.Column="1" Name="btnClose" Content="Schließen" Width="100" Background="#666666">
                <Button.Style>
                    <Style TargetType="Button" BasedOn="{StaticResource {x:Type Button}}">
                        <Setter Property="Background" Value="#666666"/>
                        <Style.Triggers>
                            <Trigger Property="IsMouseOver" Value="True">
                                <Setter Property="Background" Value="#555555"/>
                            </Trigger>
                        </Style.Triggers>
                    </Style>
                </Button.Style>
            </Button>
        </Grid>
    </Grid>
</Window>
"@

# XAML laden
$reader = [System.Xml.XmlReader]::Create([System.IO.StringReader]$xaml)
$window = [Windows.Markup.XamlReader]::Load($reader)

# UI Elemente
$btnDisableHttp2 = $window.FindName("btnDisableHttp2")
$btnUninstallUpdates = $window.FindName("btnUninstallUpdates")
$btnClose = $window.FindName("btnClose")
$txtLog = $window.FindName("txtLog")
$linkWebsite = $window.FindName("linkWebsite")
$txtUpdateStatus = $window.FindName("txtUpdateStatus")

# Logging Funktion
function Write-Log {
    param([string]$Message, [string]$Type = "Info")
    
    $timestamp = Get-Date -Format "HH:mm:ss"
    $icon = switch($Type) {
        "Success" { "✓" }
        "Error" { "✗" }
        "Warning" { "⚠" }
        default { "ℹ" }
    }
    
    $logMessage = "[$timestamp] $icon $Message`r`n"
    $txtLog.AppendText($logMessage)
    $txtLog.ScrollToEnd()
}

# Problematische Updates prüfen
function Test-ProblematicUpdates {
    Write-Log "Prüfe installierte Updates..." "Info"
    
    $updates = @("KB5066835", "KB5065789")
    $installedUpdates = @()
    
    foreach ($kb in $updates) {
        $hotfix = Get-HotFix -Id $kb -ErrorAction SilentlyContinue
        if ($hotfix) {
            $installedUpdates += $kb
            Write-Log "  → $kb ist installiert (Installiert am: $($hotfix.InstalledOn.ToString('dd.MM.yyyy')))" "Warning"
        }
        else {
            Write-Log "  → $kb ist NICHT installiert" "Success"
        }
    }
    
    Write-Log "" "Info"
    
    # UI Status aktualisieren
    if ($installedUpdates.Count -gt 0) {
        $statusText = "⚠ Gefunden: $($installedUpdates -join ', ')"
        $txtUpdateStatus.Text = $statusText
        $txtUpdateStatus.Foreground = "#D13438"
        
        Write-Log "WARNUNG: $($installedUpdates.Count) problematische(s) Update(s) gefunden!" "Warning"
        Write-Log "Diese Updates können localhost-Verbindungsprobleme verursachen." "Warning"
    }
    else {
        $txtUpdateStatus.Text = "✓ Keine problematischen Updates installiert"
        $txtUpdateStatus.Foreground = "#107C10"
        $btnUninstallUpdates.IsEnabled = $false
        
        Write-Log "Keine der problematischen Updates sind installiert." "Success"
        Write-Log "Ihr System sollte keine localhost-Verbindungsprobleme haben." "Success"
    }
    
    Write-Log "" "Info"
    
    return $installedUpdates
}

# HTTP/2 Registry-Status prüfen
function Test-Http2Status {
    Write-Log "Prüfe HTTP/2 Einstellungen..." "Info"
    
    try {
        $regPath = "HKLM:\SYSTEM\CurrentControlSet\Services\HTTP\Parameters"
        
        if (Test-Path $regPath) {
            $http2Tls = Get-ItemProperty -Path $regPath -Name "EnableHttp2Tls" -ErrorAction SilentlyContinue
            $http2Clear = Get-ItemProperty -Path $regPath -Name "EnableHttp2Cleartext" -ErrorAction SilentlyContinue
            
            if ($http2Tls.EnableHttp2Tls -eq 0 -and $http2Clear.EnableHttp2Cleartext -eq 0) {
                Write-Log "  → HTTP/2 ist bereits deaktiviert" "Success"
                return $true
            }
            elseif ($null -eq $http2Tls -and $null -eq $http2Clear) {
                Write-Log "  → HTTP/2 Einstellungen sind Standard (aktiv)" "Info"
                return $false
            }
            else {
                Write-Log "  → HTTP/2 ist teilweise konfiguriert" "Warning"
                return $false
            }
        }
        else {
            Write-Log "  → HTTP/2 Einstellungen sind Standard (aktiv)" "Info"
            return $false
        }
    }
    catch {
        Write-Log "  → Fehler beim Prüfen der HTTP/2 Einstellungen" "Warning"
        return $false
    }
}

# HTTP/2 deaktivieren
function Disable-Http2 {
    Write-Log "Starte HTTP/2 Deaktivierung..." "Info"
    
    # Aktuellen Status prüfen
    $alreadyDisabled = Test-Http2Status
    
    if ($alreadyDisabled) {
        Write-Log "HTTP/2 ist bereits deaktiviert!" "Info"
        [System.Windows.MessageBox]::Show(
            "HTTP/2 ist bereits deaktiviert.`n`nEs ist keine weitere Aktion erforderlich.",
            "Information",
            [System.Windows.MessageBoxButton]::OK,
            [System.Windows.MessageBoxImage]::Information
        )
        return
    }
    
    Write-Log "" "Info"
    
    try {
        $regPath = "HKLM:\SYSTEM\CurrentControlSet\Services\HTTP\Parameters"
        
        # Registry-Pfad erstellen falls nicht vorhanden
        if (-not (Test-Path $regPath)) {
            New-Item -Path $regPath -Force | Out-Null
            Write-Log "Registry-Pfad erstellt: $regPath" "Success"
        }
        
        # HTTP/2 TLS deaktivieren
        Set-ItemProperty -Path $regPath -Name "EnableHttp2Tls" -Value 0 -Type DWord -Force
        Write-Log "  → EnableHttp2Tls auf 0 gesetzt ✓" "Success"
        
        # HTTP/2 Cleartext deaktivieren
        Set-ItemProperty -Path $regPath -Name "EnableHttp2Cleartext" -Value 0 -Type DWord -Force
        Write-Log "  → EnableHttp2Cleartext auf 0 gesetzt ✓" "Success"
        
        Write-Log "" "Info"
        Write-Log "=== HTTP/2 ERFOLGREICH DEAKTIVIERT ===" "Success"
        Write-Log "HINWEIS: Ein Neustart wird empfohlen!" "Warning"
        
        [System.Windows.MessageBox]::Show(
            "HTTP/2 wurde erfolgreich deaktiviert!`n`nBitte starten Sie Ihren Computer neu, damit die Änderungen wirksam werden.`n`nDies sollte die localhost-Verbindungsprobleme beheben.",
            "Erfolg",
            [System.Windows.MessageBoxButton]::OK,
            [System.Windows.MessageBoxImage]::Information
        )
    }
    catch {
        Write-Log "FEHLER: $($_.Exception.Message)" "Error"
        [System.Windows.MessageBox]::Show(
            "Fehler beim Deaktivieren von HTTP/2:`n`n$($_.Exception.Message)",
            "Fehler",
            [System.Windows.MessageBoxButton]::OK,
            [System.Windows.MessageBoxImage]::Error
        )
    }
}

# Updates deinstallieren
function Uninstall-ProblematicUpdates {
    Write-Log "Starte Update-Deinstallation..." "Info"
    
    $updates = @("KB5066835", "KB5065789")
    $uninstalledCount = 0
    $notFoundCount = 0
    
    foreach ($kb in $updates) {
        Write-Log "Verarbeite Update $kb..." "Info"
        
        # Prüfen ob Update installiert ist
        $hotfix = Get-HotFix -Id $kb -ErrorAction SilentlyContinue
        
        if ($hotfix) {
            Write-Log "  → Deinstalliere $kb..." "Warning"
            
            try {
                $process = Start-Process -FilePath "wusa.exe" -ArgumentList "/uninstall /kb:$($kb.Replace('KB','')) /quiet /norestart" -Wait -PassThru
                
                if ($process.ExitCode -eq 0 -or $process.ExitCode -eq 3010) {
                    Write-Log "  → $kb erfolgreich deinstalliert ✓" "Success"
                    $uninstalledCount++
                }
                elseif ($process.ExitCode -eq -2145124329) {
                    Write-Log "  → $kb wurde nicht gefunden (bereits entfernt?)" "Info"
                    $notFoundCount++
                }
                else {
                    Write-Log "  → Fehler beim Deinstallieren von $kb (Exit Code: $($process.ExitCode))" "Error"
                }
            }
            catch {
                Write-Log "  → FEHLER bei $kb $($_.Exception.Message)" "Error"
            }
        }
        else {
            Write-Log "  → $kb ist nicht installiert" "Info"
            $notFoundCount++
        }
    }
    
    Write-Log "" "Info"
    
    if ($uninstalledCount -gt 0) {
        Write-Log "=== DEINSTALLATION ERFOLGREICH ===" "Success"
        Write-Log "$uninstalledCount Update(s) wurden entfernt." "Success"
        Write-Log "" "Info"
        
        # Update-Status aktualisieren
        $txtUpdateStatus.Text = "✓ Problematische Updates wurden entfernt"
        $txtUpdateStatus.Foreground = "#107C10"
        $btnUninstallUpdates.IsEnabled = $false
        
        $result = [System.Windows.MessageBox]::Show(
            "$uninstalledCount Update(s) wurden erfolgreich deinstalliert.`n`nMöchten Sie den Computer jetzt neu starten?`n`n(Ein Neustart ist erforderlich, damit die Änderungen wirksam werden)",
            "Neustart erforderlich",
            [System.Windows.MessageBoxButton]::YesNo,
            [System.Windows.MessageBoxImage]::Question
        )
        
        if ($result -eq [System.Windows.MessageBoxResult]::Yes) {
            Write-Log "System wird neu gestartet..." "Warning"
            Start-Sleep -Seconds 2
            Restart-Computer -Force
        }
        else {
            Write-Log "WICHTIG: Bitte starten Sie Ihren Computer manuell neu!" "Warning"
        }
    }
    elseif ($notFoundCount -eq $updates.Count) {
        Write-Log "Keine der problematischen Updates sind installiert." "Info"
        
        $txtUpdateStatus.Text = "✓ Keine problematischen Updates installiert"
        $txtUpdateStatus.Foreground = "#107C10"
        $btnUninstallUpdates.IsEnabled = $false
        
        [System.Windows.MessageBox]::Show(
            "Keine der problematischen Updates (KB5066835, KB5065789) wurden auf diesem System gefunden.`n`nEs ist keine Deinstallation erforderlich.",
            "Information",
            [System.Windows.MessageBoxButton]::OK,
            [System.Windows.MessageBoxImage]::Information
        )
    }
    else {
        Write-Log "Deinstallation teilweise fehlgeschlagen. Bitte prüfen Sie das Log." "Warning"
    }
}

# Event Handler
$btnDisableHttp2.Add_Click({
    $btnDisableHttp2.IsEnabled = $false
    $btnUninstallUpdates.IsEnabled = $false
    
    Disable-Http2
    
    $btnDisableHttp2.IsEnabled = $true
    $btnUninstallUpdates.IsEnabled = $true
})

$btnUninstallUpdates.Add_Click({
    $result = [System.Windows.MessageBox]::Show(
        "Sind Sie sicher, dass Sie die Updates KB5066835 und KB5065789 deinstallieren möchten?`n`nDies kann einige Minuten dauern und erfordert einen Neustart.",
        "Bestätigung",
        [System.Windows.MessageBoxButton]::YesNo,
        [System.Windows.MessageBoxImage]::Warning
    )
    
    if ($result -eq [System.Windows.MessageBoxResult]::Yes) {
        $btnDisableHttp2.IsEnabled = $false
        $btnUninstallUpdates.IsEnabled = $false
        
        Uninstall-ProblematicUpdates
        
        $btnDisableHttp2.IsEnabled = $true
        $btnUninstallUpdates.IsEnabled = $true
    }
})

$btnClose.Add_Click({
    $window.Close()
})

$linkWebsite.Add_RequestNavigate({
    param($sender, $e)
    Start-Process $e.Uri.AbsoluteUri
    $e.Handled = $true
})

# Initialisierung
Write-Log "=== Localhost Connection Repair Tool ===" "Info"
Write-Log "Version 1.0 - Windows 11" "Info"
Write-Log "Administrator-Rechte: Aktiv ✓" "Success"
Write-Log "" "Info"

# System-Diagnose durchführen
Write-Log "=== SYSTEM-DIAGNOSE ===" "Info"
Write-Log "" "Info"

# Updates prüfen
$installedProblematicUpdates = Test-ProblematicUpdates

# HTTP/2 Status prüfen
$http2Disabled = Test-Http2Status

Write-Log "=== DIAGNOSE ABGESCHLOSSEN ===" "Info"
Write-Log "" "Info"

# Empfehlung ausgeben
if ($installedProblematicUpdates.Count -gt 0) {
    if ($http2Disabled) {
        Write-Log "EMPFEHLUNG: HTTP/2 ist bereits deaktiviert. Sie können die Updates deinstallieren, um HTTP/2 wieder zu aktivieren." "Info"
    }
    else {
        Write-Log "EMPFEHLUNG: Deinstallieren Sie die problematischen Updates für eine dauerhafte Lösung." "Warning"
    }
}
else {
    Write-Log "Ihr System ist nicht von den bekannten localhost-Problemen betroffen." "Success"
}

Write-Log "" "Info"
Write-Log "Bitte wählen Sie bei Bedarf eine Reparatur-Option aus." "Info"

# Fenster anzeigen
$window.ShowDialog() | Out-Null